<?php

namespace wcf\system\html\output\node;

use wcf\util\DOMUtil;

/**
 * Normalizes HTML generated by earlier version of WoltLab Suite.
 *
 * @author Alexander Ebert
 * @copyright 2001-2023 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since 6.0
 */
final class HtmlOutputNodeNormalizer
{
    public function normalize(\DOMXPath $xpath): void
    {
        $this->normalizeBr($xpath);

        $candidates = $this->getPossibleSpacerParagraphs($xpath);
        $this->reduceSpacerParagraphs($candidates);

        $this->stripMarkerOnBr($xpath);

        $this->reduceSubsequentHr($xpath);
        $this->reduceBrInEmptyParagraph($xpath);
        $this->reduceSubsequentEmptyParagraphs($xpath);
        $this->reduceSubsequentBr($xpath);
    }

    /**
     * @return list<\DOMElement>
     */
    private function getPossibleSpacerParagraphs(\DOMXpath $xpath): array
    {
        $paragraphs = [];

        foreach ($xpath->query('//p') as $p) {
            \assert($p instanceof \DOMElement);

            if ($p->childNodes->length === 1) {
                $child = $p->childNodes->item(0);
                if ($child->nodeName === 'br') {
                    \assert($child instanceof \DOMElement);

                    if ($child->getAttribute('data-cke-filler') !== 'true') {
                        $paragraphs[] = $p;
                    }
                }
            }
        }

        return $paragraphs;
    }

    /**
     * @param list<\DOMElement> $paragraphs
     * @return void
     */
    private function reduceSpacerParagraphs(array $paragraphs): void
    {
        if ($paragraphs === []) {
            return;
        }

        for ($i = 0, $length = \count($paragraphs); $i < $length; $i++) {
            $candidate = $paragraphs[$i];
            $offset = 0;

            // Searches for adjacent paragraphs.
            while ($i + $offset + 1 < $length) {
                $nextCandidate = $paragraphs[$i + $offset + 1];
                if ($candidate->nextElementSibling !== $nextCandidate) {
                    break;
                }

                $offset++;
            }

            if ($offset === 0) {
                // An offset of 0 means that this is a single paragraph and we
                // can safely remove it.
                $candidate->remove();
            } else {
                // We need to reduce the number of paragraphs by half, unless it
                // is an uneven number in which case we need to remove one
                // additional paragraph.
                $totalNumberOfParagraphs = $offset + 1;
                $numberOfParagraphsToRemove = \ceil($totalNumberOfParagraphs / 2);

                $removeParagraphs = \array_slice($paragraphs, $i, $numberOfParagraphsToRemove);
                foreach ($removeParagraphs as $paragraph) {
                    $paragraph->remove();
                }

                $i += $offset;
            }
        }
    }

    private function normalizeBr(\DOMXpath $xpath): void
    {
        foreach ($xpath->query('//br') as $br) {
            \assert($br instanceof \DOMElement);

            $this->unwrapBr($br);
            $this->removeTrailingBr($br);
        }
    }

    private function unwrapBr(\DOMElement $br): void
    {
        for (;;) {
            if ($br->previousSibling || $br->nextSibling) {
                return;
            }

            $parent = $br->parentNode;
            switch ($parent->nodeName) {
                case "b":
                case "del":
                case "em":
                case "i":
                case "strong":
                case "sub":
                case "sup":
                case "span":
                case "u":
                    $parent->parentNode->insertBefore($br, $parent);
                    $parent->parentNode->removeChild($parent);
                    break;

                default:
                    return;
            }
        }
    }

    private function removeTrailingBr(\DOMElement $br): void
    {
        $paragraphOrTableCell = DOMUtil::closest($br, "p");
        if ($paragraphOrTableCell === null) {
            $paragraphOrTableCell = DOMUtil::closest($br, "td");

            if ($paragraphOrTableCell === null) {
                return;
            }
        }

        if (!DOMUtil::isLastNode($br, $paragraphOrTableCell)) {
            return;
        }

        if ($paragraphOrTableCell->nodeName === "td" || $paragraphOrTableCell->childNodes->length > 1) {
            $br->remove();
        }
    }

    private function stripMarkerOnBr(\DOMXPath $xpath): void
    {
        foreach ($xpath->query(".//br[@data-cke-filler='true']") as $br) {
            $br->removeAttribute("data-cke-filler");
        }
    }

    /**
     * Reduces multiple subsequent <hr> into a single <hr>.
     */
    private function reduceSubsequentHr(\DOMXPath $xpath): void
    {
        /** @var list<\DOMElement> */
        $elements = [];
        foreach ($xpath->query(".//hr") as $hr) {
            \assert($hr instanceof \DOMElement);
            $elements[] = $hr;
        }

        for ($i = 0, $length = \count($elements); $i < $length; $i++) {
            $hr = $elements[$i];

            /** @var list<\DOMElement> */
            $foundHrs = [];
            while ($i + 1 < $length) {
                if ($hr->nextElementSibling === $elements[$i + 1]) {
                    $superfluousHr = $elements[$i + 1];
                    $foundHrs[] = $superfluousHr;
                    $hr = $superfluousHr;

                    $i++;
                } else {
                    break;
                }
            }

            if ($foundHrs !== []) {
                foreach ($foundHrs as $hr) {
                    $hr->remove();
                }
            }
        }
    }

    /**
     * Reduce multiple <br> inside a <p> if they are the only children.
     */
    private function reduceBrInEmptyParagraph(\DOMXPath $xpath): void
    {
        foreach ($xpath->query(".//p") as $paragraph) {
            \assert($paragraph instanceof \DOMElement);

            $foundBrs = [];
            foreach ($paragraph->childNodes as $child) {
                if ($child instanceof \DOMElement && $child->tagName === 'br') {
                    $foundBrs[] = $child;
                } else {
                    // Check the next paragraph.
                    continue 2;
                }
            }

            $length = \count($foundBrs);
            if ($length > 1) {
                for ($i = 1; $i < $length; $i++) {
                    $br = $foundBrs[$i];
                    \assert($br instanceof \DOMElement);

                    $br->remove();
                }
            }
        }
    }

    /**
     * Reduces multiple empty paragraphs into a single one.
     */
    private function reduceSubsequentEmptyParagraphs(\DOMXPath $xpath): void
    {
        /** @var list<\DOMElement> */
        $elements = [];
        foreach ($xpath->query(".//p") as $paragraph) {
            \assert($paragraph instanceof \DOMElement);

            $elements[] = $paragraph;
        }

        for ($i = 0, $length = \count($elements); $i < $length; $i++) {
            $paragraph = $elements[$i];

            if ($paragraph->childNodes->length > 1 || $paragraph->childElementCount !== 1) {
                continue;
            }

            if ($paragraph->firstElementChild->tagName !== "br") {
                continue;
            }

            $subsequentParagraphs = [$paragraph];
            while ($i + 1 < $length) {
                $nextParagraph = $elements[$i + 1];

                if ($paragraph->nextElementSibling !== $nextParagraph) {
                    break;
                }

                if ($nextParagraph->childNodes->length > 1 || $nextParagraph->childElementCount !== 1) {
                    break;
                }

                if ($nextParagraph->firstElementChild->tagName !== "br") {
                    break;
                }

                $subsequentParagraphs[] = $nextParagraph;
                $paragraph = $nextParagraph;
                $i++;
            }

            $foundParagraphs = \count($subsequentParagraphs);
            if ($foundParagraphs > 1) {
                for ($j = 1; $j < $foundParagraphs; $j++) {
                    $subsequentParagraphs[$j]->remove();
                }
            }
        }
    }

    /**
     * Reduces more than three subsequent <br>.
     */
    private function reduceSubsequentBr(\DOMXPath $xpath): void
    {
        /** @var list<\DOMElement> */
        $elements = [];
        foreach ($xpath->query(".//br") as $br) {
            \assert($br instanceof \DOMElement);
            $elements[] = $br;
        }

        for ($i = 0, $length = \count($elements); $i < $length; $i++) {
            $br = $elements[$i];

            $chainedBrs = [$br];
            while ($i + 1 < $length) {
                $nextBr = $elements[$i + 1];

                if ($br->nextSibling === $nextBr) {
                    $chainedBrs[] = $nextBr;
                    $br = $nextBr;

                    $i++;
                } else {
                    break;
                }
            }

            $foundBrs = \count($chainedBrs);
            if ($foundBrs > 3) {
                for ($j = 3; $j < $foundBrs; $j++) {
                    $chainedBrs[$j]->remove();
                }
            }
        }
    }
}
