<?php

namespace wcf\system\cache\source;

use wcf\system\io\AtomicWriter;
use wcf\system\WCF;

/**
 * DiskCacheSource is an implementation of CacheSource that stores the cache as simple files in the file system.
 *
 * @author  Alexander Ebert, Marcel Werk
 * @copyright   2001-2024 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 */
final class DiskCacheSource implements ICacheSource
{
    /**
     * @inheritDoc
     */
    public function flush($cacheName, $useWildcard)
    {
        if ($useWildcard) {
            $quoted = \preg_quote($cacheName, '/');
            $regex = "/^cache\\.{$quoted}(-[a-f0-9]+)?\\.v2\\.php$/";

            $iterator = new \DirectoryIterator(WCF_DIR . 'cache/');
            foreach ($iterator as $file) {
                if (!$file->isFile()) {
                    continue;
                }
                if ($file->getExtension() !== 'php') {
                    continue;
                }
                if (!\preg_match($regex, $file->getBasename())) {
                    continue;
                }

                WCF::resetZendOpcache($file->getPathname());

                if (!@\touch($file->getPathname(), 1)) {
                    \unlink($file->getPathname());
                }
            }
        } else {
            $filename = $this->getFilename($cacheName);

            if (\file_exists($filename)) {
                WCF::resetZendOpcache($filename);

                if (!@\touch($filename, 1)) {
                    \unlink($filename);
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function flushAll()
    {
        $iterator = new \DirectoryIterator(WCF_DIR . 'cache/');
        foreach ($iterator as $file) {
            if (!$file->isFile()) {
                continue;
            }
            if ($file->getExtension() !== 'php') {
                continue;
            }

            \unlink($file->getPathname());
        }

        WCF::resetZendOpcache();
    }

    /**
     * @inheritDoc
     */
    public function get($cacheName, $maxLifetime)
    {
        $filename = $this->getFilename($cacheName);
        if ($this->needRebuild($filename, $maxLifetime)) {
            return;
        }

        // load cache
        try {
            return $this->readCache($filename);
        } catch (\Exception $e) {
            return;
        }
    }

    /**
     * @inheritDoc
     */
    public function set($cacheName, $value, $maxLifetime)
    {
        $writer = new AtomicWriter($this->getFilename($cacheName));
        $writer->write("<?php /* cache: " . $cacheName . " (generated at " . \gmdate('r') . ") DO NOT EDIT THIS FILE */ return <<<'EOT'\n");
        $writer->write(\serialize($value));
        $writer->write("\nEOT;\n");
        $writer->flush();
        $writer->close();

        WCF::resetZendOpcache($this->getFilename($cacheName));
    }

    /**
     * Returns cache filename.
     */
    private function getFilename(string $cacheName): string
    {
        return WCF_DIR . 'cache/cache.' . $cacheName . '.v2.php';
    }

    /**
     * Determines whether the cache needs to be rebuild or not.
     */
    private function needRebuild(string $filename, int $maxLifetime): bool
    {
        // cache does not exist
        if (!\file_exists($filename)) {
            return true;
        }

        // cache is empty
        if (!@\filesize($filename)) {
            return true;
        }

        // cache resource was marked as obsolete
        if (($mtime = \filemtime($filename)) <= 1) {
            return true;
        }

        // maxlifetime expired
        if ($maxLifetime > 0 && (TIME_NOW - $mtime) > $maxLifetime) {
            return true;
        }

        // do not rebuild cache
        return false;
    }

    /**
     * Loads the file of a cached resource.
     */
    private function readCache(string $filename): mixed
    {
        $contents = include($filename);

        try {
            return \unserialize($contents);
        } catch (\Throwable $e) {
            throw new \Exception("Failed to unserialize the cache contents.", 0, $e);
        }
    }
}
